function H = paleyI(N)

% Computes Hadamard matrices (H-matrices) of Paley type I 
%
% Usage: 
% H = paleyI(N);
%
% N: order of the Hadamard matrix to be computed (integer)
% H: returns the computed Hadamard matrix or
%    returns 0 if an error occurred (invalid matrix order or incorrect H-matrix)
%   
%
% H-matrices of Paley type I are defined for orders N = 4m = p+1 (m=1,2,3,4,...),
% where p is a prime with mod(p,4)=3, i.e. N=12, 20, 44, 60, 68, 72, 84, ... 
% (omitting valid Paley type I matrix orders that are equal to N=2^n (n=1,2,3,...))
% [1,2]. 
% 
% H-matrices contain only '+1' and '-1' elements and are orthogonal. 
% 
% The first row and column of a Paley type I H-matrix consists only of '+1' 
% elements. All diagonal elements H(l,m) with l=m~=1 are equal to '-1'. The 
% determination of all other elements is based on evaluating the Legendre symbol, 
% which is defined to be equal to '+1' or '-1' if its argument is a quadratic residue 
% or a quadratic non-residue.
%
% [1] R. E. A. C. Paley, 'On Orthogonal Matrices', J. Math. Phys., vol. 12, 
%     pp. 311-320, (1933)
% [2] X. Yang and Z. Wen and Y. Xu, 'Construction of Hadamard phase-codes 
%     for holographic memories', Proc. SPIE, vol. 2849, pp. 217-228, (1996)
%
%
% Gernot Berger (C) 2008


% Get largest prime number (<N)
p = primes(N-1);
p = p(end);

% Check whether N is a correct Paley type I matrix order
if (mod(N-1,4) == 3) && (p == N-1)

    % Determine quadratic residues
    qres = mod([1:N-2].^2, (N-1));
    
    % Compute second row elements H(2,2),H(2,3),H(2,4),...,H(2,N) of the final H-matrix
    X = ones(1,N-1);
    X(1,1) = -1;
    for n = 2:(N-1)
        if (n<p+1) && isempty(find(qres == (n-1)))
            X(1,n) = -1;
        end;
    end;
    
    H = ones(N,N);
    
    % Copy the above computed row to form the right and left triangular matrices
    for n = 1:N-1
        H(n+1,n+1:N) = X(1:N-n);
        H(N-n+1,2:N-n) = X(n+1:N-1);
    end;

    % Check whether the computed matrix is a correct Hadamard matrix 
    % (This test may be omitted. When I tested the function for every valid 
    % matrix order N<=5000 all generated matrices were correct H-matrices.) 
    if ((H.' * H)/N) ~= eye(N)
        disp(['H_', num2str(N),' is not a correct Hadamard matrix.']);
        H = 0;
    end;
    
else

    disp(['N = ', num2str(N),' is not a valid Paley type I matrix order.']);
    H = 0;
    
end;
