function [img, newimg] = prepostLTdownup(imgname,prepostlth,bioScale,rfactor,dopre,dopost)
% Function to demonstrate performance of pre- and post-filtering 
% in DCT-based down- and up-sampling.
%
% Usage:
%  [img, newimg]=prepostLTdownup(imgname,prepostlth,bioScale,rfactor,dopre,dopost):
%
%    imgname: Name of the input image;
%    prepostlth: Length of pre and post filters: 2, 4, 6, or 8;
%    bioScale: Scaling coefficient for biorthogonal filters: 
%              1 for orthogonal,  8/5 25/16 23/16 3/2 sqrt(2) etc for biorthogonal;
%    rfactor: Resize factor
%    dopre:   1/0: with/without pre processing;
%    dopost:  1/0: with/without post processing;
%
%    img :    Original image data;
%    newimg:  The image data after all processing;
%
% Example:
%    [img, rec_img]=prepostLTdownup('barb2.tif', 8, 23/16, 1/2, 1, 1);
%
% Trac D. Tran
% Department of Electrical and Computer Engineering
% The Johns Hopkins University
% 02/28/2006

if nargin ~= 6
    disp('Type ''help prepostLTdownup'' for usage.');
    break;
end

%%%%%%%%%%%%%%%%%%%%%%%%%%
% Read image
%%%%%%%%%%%%%%%%%%%%%%%%%%

img=imread(imgname);
img = double(img);
[row,col] = size(img);
rowblk = row/8; colblk = col/8;
% Keep original image for PSNR comparison
newimg = img;

%%%%%%%%%%%%%%%%%%%%
% Generate pre and post filters:
%%%%%%%%%%%%%%%%%%%%
N = prepostlth; N2 = N/2;
IN2 = eye(N2); JN2 = fliplr(IN2); ZN2 = zeros(N2);
dctIV = cos(pi*[1:2:(2*N2-1)]'*[1:2:(2*N2-1)]/(4*N2)) * sqrt(2/N2);
scale = diag([bioScale ones(1, N2-1)]);
%The lower branch consists of IDCT-II, scaling, and DCT-IV:
V= [ 0.9454    0.7917    0.4207    0.3680;     
    -0.5654    0.8863    0.6731    0.3630;
     0.1118   -0.3891    1.1034    0.5055;
    -0.0312    0.0033   -0.1386    1.2449]; %2-VM solution
V = JN2 * dct(IN2)' * scale * dctIV * JN2;
Bf=[IN2/2 JN2/2;JN2 -IN2]; % forward Haar
Bi=[IN2 JN2/2;JN2 -IN2/2]; % inverse Haar
preflt = Bi * [IN2, ZN2; ZN2, V] * Bf;
postflt = inv(preflt);

%%%%%%%%%%%%%%%%%%%%
% preprocessing
%%%%%%%%%%%%%%%%%%%%
if dopre == 0
  disp('Pre-processing is skipped.');
else
   %col transform:
   for blk = 1 :  rowblk - 1
      %Prefiltering the boundary of all columns:
      %The rows are: ref --> ref + N-1
      %Ref is the first pixel covered by the prefilter at this blk boundary:
      ref = blk * 8 - (N2-1);
      newimg(ref : ref + (N-1), 1 : col) = preflt * newimg(ref : ref + (N-1), 1 : col);
   end

   %row transform:
   for blk = 1 :  colblk - 1
      %Prefiltering the boundary of all rows, need transposes:
      ref = blk * 8 - (N2-1);
      newimg(1 : row, ref : ref + (N-1)) = (preflt * newimg(1 : row, ref : ref + (N-1))' )';
   end
   
   %Check out the effect of pre-processing
   figure; imshow(newimg/256,'truesize');
   title('After preprocessing');
   
   %Check overflow and underflow
   under=sum(sum(newimg<0));
   s=sprintf('Number of underflow pixels after pre-processing: %d', under);
   disp(s);
   over=sum(sum(newimg>255));
   s=sprintf('Number of overflow pixels after pre-processing: %d', over);
   disp(s);
   %figure;imshow((newimg<0 | newimg>255),'truesize');title('Out-of-range Map');
end

%newimg2=img;
%C=preflt; 
%fun = @prepostfiltering;
%newimg2(:,5:col-4)=blkproc(newimg2(:,5:col-4),[1 8],fun,C);
%newimg2(5:row-4,:)=blkproc(newimg2(5:row-4,:),[8 1],fun,C');
%figure; imshow(abs(newimg-newimg2)/max(max(abs(newimg-newimg2))),'truesize');

%%% Down-sampling via DCT %%%
C=dct(eye(8)); 
fun = @blockDCT;
y=blkproc(newimg,[8 8],fun,C,round(rfactor*8));

%%% Up-sampling via DCT %%%
fun = @blockIDCT;
newimg=blkproc(y,[8 8],fun,C,round(rfactor*8));

%%%%%%%%%%%%%%%%%%%%
% Postprocessing
%%%%%%%%%%%%%%%%%%%%
if dopost == 0
   disp('Post-processing is skipped.');
else
   % row transform:
   for blk = 1 :  colblk - 1
      ref = blk * 8 - (N2-1);
      newimg(1 : row, ref : ref + (N-1)) = (postflt * newimg(1 : row, ref : ref + (N-1))' )';
   end
   
   % col transform:
   for blk = 1 :  rowblk - 1
      ref = blk * 8 - (N2-1);
      newimg(ref : ref + (N-1), 1 : col) = postflt * newimg(ref : ref + (N-1), 1 : col);
   end
end

newimg=double(uint8(newimg+1/2)); %need this to compute PSNR
figure; imshow(newimg/256,'truesize');
title(['Reconstructed Image    Pre/Post-Processing Length = ',...
        num2str(prepostlth),'    PSNR = ',num2str(imgpsnr(img, newimg)),' dB']);
  
clear all;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%  End
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function psnr = imgpsnr(img1, img2)
% Calculate the PSNR and max error between two images
err= abs(img1 - img2);
maxerr = max(max(err));
s=sprintf('Max error: %d', maxerr);disp(s);
err = err .^ 2;
mse = sum(sum(err)) / size(img1,1) / size(img1,2);
psnr = 10 * log10(255*255/mse);
s=sprintf('PSNR:%f dB',psnr);disp(s);

function y=blockDCT(x,C,r);
% y=blockDCT(x,C,r);
% Perform transformation via C & retain coeffs via r
y=C*x*C'; %Compute the DCT coefficients y
Q=zeros(8,8); Q(1:r,1:r)=ones(r,r);
y=y.*Q;

function xr=blockIDCT(y,C,q);
% xr=blockIDCT(y,C,q);
% Perform inverse quantization with stepsize q & inverse transform via C 
% on an 8x8 transform coefficient block y
xr=C'*y*C;

function y=prepostfiltering(x,P);
% y=prepostfiltering(x,P);
y=x*P';