%% Outrunner BLDC Motor Size Optimization
%% David Meeker
%% 01Feb2015

openfemm

%% Initial Geometry Guess
rso = 10; 		% Stator outer radius 
rsi = 2.35; 	% Stator inner radius 
dm = 1; 		% Magnet thickness 
dc = 1; 		% Can thickness 
ds = 0.5; 		% depth of slot opening 
fm = 0.860533; 	% Pole fraction spanned by the magnet 
fp = 0.7; 		% Pole fraction spanned by the iron 
ft = 0.95; 		% Width of tooth as a fraction of pole pitch at stator ID 
fb = 1; 		% Back iron thickness as a fraction of tooth thickness 
go = 0.5; 		% stator to magnet mechanical clearance 
hh = 25; 		% length in the into-the-page direction 
Jpk = 10.0; 	% peak current density in the winding 

%% Desired Torque
tqdes = 1.0; 

%% Set up various parameters internal to the optimization
nmax = 1000;		% Maximum number of iterations in the optimization
nstall = 20;		% Number of iterations before a stall is declared
d = 0.1;			% Starting step size
dmin = 0.01;		% Step size at which it is assumed that we've converged

bestVolume = 0;		% Place holder for optimal cost 
stall = 0;			% Number of iterations since the last forward progress

for k = 1:nmax
	disp(sprintf('Iteration %i; d = %f; stall = %i',k,d,stall));
	
	bOK = 0;
	while (bOK == 0) 
	
		% If first time through the loop, evaluate the initial condition
		% Otherwise, randomly pick a nearby geometry
		if (k == 1)
			dd = 0;
		else
			dd = d;
		end;
		
		% Randomly pick a new candidate geometry
		newrso = rso* (1 + dd*myrand);
		newrsi = rsi* (1 + dd*myrand);
		newdm  = dm * (1 + dd*myrand); 
		newds  = ds * (1 + dd*myrand); 
		newdc  = dc * (1 + dd*myrand);
		newfm  = fm * (1 + dd*myrand); 
		newfp  = fp * (1 + dd*myrand);
		newft  = ft * (1 + dd*myrand); 
		newfb  = fb * (1 + dd*myrand);
		newhh  = hh;
		
		bOK=1;
		
		% Check to make sure the candidate geometry isn't bad
		if ((newrsi + newds) > newrso) bOK=0; end; 
		if (newds < 0) bOK=0; end;
		if (newdc < 0) bOK=0; end;
		if ((newfm > 1) || (newfm<0)) bOK=0; end;
		if ((newfp > 1) || (newfp<0)) bOK=0; end;
		if ((newft > 1) || (newft<0)) bOK=0; end;
		
		% Check on any other constraints.
		% If your constraint isn't met, just say the geometry isn't OK...
		newrro = (newrso + go + newdm + newdc); % rotor outer radius
%		if ((2*newrro) > 25) bOK=0; end; % Example constraint on motor OD
	end;
	
	% Build and analyze candidate geometry
	BuildMotor(newrso, newrsi, newdm, newdc, newds, newfm, newfp, newft, newfb, go, hh, Jpk);
	mi_saveas('temp.fem');
	mi_analyze(1);
	mi_loadsolution;
	
	% Compute torque for a fixed length to figure out how long the machine
	% needs to be to get the desired torque;
	mo_groupselectblock(1); 
	tq = mo_blockintegral(22);
	newhh = hh*tqdes/abs(tq);
	
	% Build cost function.  Here, the cost function is taken to be a machine size metric
	newlen = newhh + rso/2; % core + guesstimated end turn build
	thisVolume =  newlen*pi*(newrro^2 - newrsi^2)/1000; % scaled so thisVolume is in units of cm^3
	
	% See if this candidate is better than the previous optimum.
	% If so, this candidate is the new optimum
	stall = stall + 1;
	if ((thisVolume < bestVolume) || (k==1))
		stall = 0;
		bestVolume = thisVolume;
		hh  = newhh;
		rso = newrso;
		rsi = newrsi;
		dm  = newdm;
		ds  = newds;
		dc  = newdc;
		fm  = newfm;
		fp  = newfp;
		ft  = newft;
		fb  = newfb;
		disp(sprintf('bestVolume = %f; rro = %f; rso = %f; hh = %f',bestVolume, newrro, rso, hh));
	end;
	
	% Run through the 'stall logic' to see if the step size should be reduced.
	if (stall > nstall)
		d = d/2;
		stall = 0;
		if (d < dmin)
			break;
		end;
	end;
	
	% clean up before next iteration
	mo_close
	mi_close
	
end;

%% Finished! Report the results
disp(sprintf('Optimal volume = %f',bestVolume));
disp(sprintf('rso = %f',rso));
disp(sprintf('rsi = %f',rsi));
disp(sprintf('dm  = %f', dm));
disp(sprintf('dc  = %f', dc));
disp(sprintf('ds  = %f', ds));
disp(sprintf('fm  = %f', fm));
disp(sprintf('fp  = %f', fp));
disp(sprintf('ft  = %f', ft));
disp(sprintf('fb  = %f', fb));
disp(sprintf('go  = %f', go));
disp(sprintf('hh  = %f', hh));
disp(sprintf('Jpk = %f', Jpk));

closefemm
